define([
    'jquery',
    'underscore',
    'backbone',
    'modules/new-appointment-request/view',
    'modules/notification-preferences/notification-preferences-module',
    'modules/appointments/appointments-radio',
    'modules/appointments/messages/request-message',
    'modules/new-appointment-request/tests/helpers/general-helper',
], function($, _, Backbone, View, notificationModule, radio, MessageModel, helpers) {
    'use strict';

    describe('New Appointment View', function() {
        var region;
        var view;
        var gas;

        beforeEach(function() {
            region = helpers.setupTestingRegion();
            view = new View({
                googleAnalyticsProvider: {
                    getInstance: function() {
                        gas = jasmine.createSpyObj('gas', ['gas']);
                        return gas;
                    },
                },
                schedulingStepsLayoutProvider: {
                    getInstance: function() {
                        return new Backbone.View();
                    },
                },
            });
            region.show(view);
        });


        describe('Header', function() {
            var header;

            beforeEach(function() {
                header = view.getRegion('header').currentView;
            });

            it('creates the correct title', function() {
                var title = header.$('h2').text();
                expect(title).toBe('New Appointment/Request');
            });

            it('has a back button', function() {
                var button = header.$('#back-btn');
                expect(button.length).toBe(1);
                expect(button.text()).toBe('Appointments/Requests');
            });
        });


        describe('Cancel Button Clicked', function() {
            beforeEach(function() {
                spyOn(view, 'disableShowConfirmNavigation').and.callFake(_.noop);
                spyOn(view, 'routeHome').and.callFake(_.noop);
            });


            describe('Before Scheduling Method Selected', function() {
                it('navigates right away on click', function() {
                    view.onCancel();
                    expect(view.disableShowConfirmNavigation).toHaveBeenCalled();
                    expect(view.routeHome).toHaveBeenCalled();
                });
            });


            describe('After Scheduling Method Selected', function() {
                beforeEach(function() {
                    view.model.set('scheduling-method', 'direct', {silent: true});
                    view.onCancel();
                });

                it('navigates right away on click', function() {
                    expect(view.disableShowConfirmNavigation).not.toHaveBeenCalled();
                    expect(view.routeHome).not.toHaveBeenCalled();
                });

                it('creates a popup model', function(done) {
                    var popup = $('#cancel-confirmation-popup');
                    setTimeout(function() {
                        var title = popup.find('h3').text()
                            .trim();
                        var content = popup.find('p').text()
                            .trim();
                        var yesButton = popup.find('#cancel-continue-btn');
                        var noButton = popup.find('#cancel-return-btn');

                        expect(title).toBe('Cancel Confirmation');
                        expect(content).toBe('Are you sure you want to cancel scheduling this appointment?');
                        expect(yesButton.text()).toBe('Yes');
                        expect(noButton.text()).toBe('No');
                        done();
                    }, 800);
                });
            });
        });


        describe('Email View', function() {
            it('shows an empty email', function() {
                var radioInput;
                
                view.showEmailRegion();
                radioInput = view.$('input[name=emailAllowed]:checked');
                expect(radioInput.val()).toBe('false');
            });

            it('shows filled field', function() {
                var radioInput;
                var input;
                
                notificationModule.notificationPreferences.set({
                    emailAllowed: true,
                    emailAddress: 'test@test.com',
                });
                view.showEmailRegion();
                radioInput = view.$('input[name=emailAllowed]:checked');
                input = view.$('#emailAddress');

                expect(radioInput.val()).toBe('true');
                expect(input.val()).toBe('test@test.com');
            });

            it('shows a different email form for express care', function() {
                var toc = {
                    'id': 'CR1',
                    'name': 'Express Care',
                    'objectType': 'TypesOfCare',
                    'link': [],
                };
                var input;
                var text;
                var expected;
                
                view.model.set('typeOfCare', new Backbone.Model(toc));

                notificationModule.notificationPreferences.set({
                    emailAllowed: true,
                    emailAddress: 'test@test.com',
                });
                view.showEmailRegion();
                input = view.$('#emailAddress');
                text = view.getRegion('email').currentView.$('p').text();
                expected = 'Express Care visits require an email address so that we may email or text you about your care visit.\n' +
                    'Please confirm your notification contact information before submitting this request.';

                expect(input.val()).toBe('test@test.com');
                expect(text.trim()).toBe(expected.trim());
            });

            it('shows a different email form for video visit', function() {
                var clinic = {secondaryStopCode: '179'};
                var input;
                var text;
                var expected;
                
                view.model.set('clinic', new Backbone.Model(clinic));
                notificationModule.notificationPreferences.set({
                    emailAllowed: true,
                    emailAddress: 'test@test.com',
                });
                view.showEmailRegion();
                input = view.$('#emailAddress');
                text = view.getRegion('email').currentView.$('p').text();
                expected = 'Video Visits require that you accept Appointment and Request notifications, and that you provide an email address so that we may send you the video connection information.\n' +
                    'Please confirm your notification contact information before scheduling this appointment.';

                expect(input.val()).toBe('test@test.com');
                expect(text.trim()).toBe(expected.trim());
            });
        });


        describe('Save email', function() {
            var attributes;
            var options;

            function mockSave(a, o) {
                attributes = a;
                options = o;
            }

            beforeEach(function() {
                notificationModule.notificationPreferences.set({
                    emailAllowed: true,
                    emailAddress: 'test@test.com',
                });
                view.emailModel = notificationModule.notificationPreferences;
                spyOn(notificationModule.notificationPreferences, 'save').and.callFake(mockSave);
            });

            it('creates the correct save data', function() {
                view.saveEmailPreferences();

                expect(attributes.emailAllowed).toBeTruthy();
                expect(attributes.emailAddress).toBe('test@test.com');
                expect(attributes.notificationFrequency).toBe('Each new message');
                expect(options.type).toBe('PUT');
                expect(options.url).toBe('../VeteranAppointmentRequestService/v4/rest/patient/assigning-authority/testpatient/preference');
            });
        });

        // TODO UNIT TESTS SHOULD NOT HAVE CONDITIONS FOR IF THEY RUN
        if (_.isUndefined(helpers.getDisabledFeatures().get('TH_FEATURE_SET'))) {
            // Telehealth unit test changes
            describe('Submit Button Name', function() {
                it('returns submit requests for request scheduling', function() {
                    var requestLabel = 'Submit Request';

                    view.model.set('scheduling-method', 'clerk', {silent: true});
                    expect(view.getSubmitButtonName()).toBe(requestLabel);
                });

                it('return Confirm Appointment for direct scheduling', function() {
                    var confirmLabel = 'Confirm Appointment';

                    view.model.set('scheduling-method', 'direct', {silent: true});
                    expect(view.getSubmitButtonName()).toBe(confirmLabel);
                });
            });

            describe('On Scheduling Method Change', function() {
                beforeEach(function() {
                    spyOn(view, 'showEmailRegion').and.callFake(_.noop);
                });

                it('shows the request submit button', function() {
                    var button;
                    
                    view.model.set('scheduling-method', 'clerk', {silent: true});
                    button = view.ui.submit[0];

                    // Pre-Check that the view is in the right state
                    expect(button.classList.contains('hidden')).toBeTruthy();

                    view.onMethodChange();

                    expect(button.classList.contains('hidden')).toBeFalsy();
                    expect(button.innerText).toBe('Submit Request');
                });

                it('should call Google Analytics when the scheduling method is switched to "Submit a Request to a VA Scheduler"', function() {
                    view.model.set('scheduling-method', 'clerk', {silent: true});
                    view.onMethodChange();

                    expect(gas.gas).toHaveBeenCalledWith('send', 'event', 'veteran-appointment', 'user-selected-scheduling-method', 'Submit a Request to a VA Scheduler');
                });

                it('shows the direct confirm button', function() {
                    var button;
                    
                    view.model.set('scheduling-method', 'direct', {silent: true});
                    button = view.ui.confirm[0];

                    // Pre-Check that the view is in the right state
                    expect(button.classList.contains('hidden')).toBeTruthy();

                    view.onMethodChange();

                    expect(button.classList.contains('hidden')).toBeFalsy();
                    expect(button.innerText).toBe('Confirm Appointment');
                });

                it('should call Google Analytics when the scheduling method is switched to "Schedule Myself"', function() {
                    view.model.set('scheduling-method', 'direct', {silent: true});
                    view.onMethodChange();
                    
                    expect(gas.gas).toHaveBeenCalledWith('send', 'event', 'veteran-appointment', 'user-selected-scheduling-method', 'Schedule Myself');
                });
            });
        } else {
            describe('Submit Button Name', function() {
                it('returns submit requests for request scheduling', function() {
                    var requestLabel = 'Submit Request';

                    view.model.set('scheduling-method', 'clerk', {silent: true});
                    expect(view.getSubmitButtonName()).toBe(requestLabel);
                });

                it('return schedule appointments for direct scheduling', function() {
                    var directLabel = 'Schedule Appointment';

                    view.model.set('scheduling-method', 'direct', {silent: true});
                    expect(view.getSubmitButtonName()).toBe(directLabel);
                });
            });
            describe('On Scheduling Method Change', function() {
                beforeEach(function() {
                    spyOn(view, 'showEmailRegion').and.callFake(_.noop);
                });

                it('shows the request submit button', function() {
                    var button;
                    
                    view.model.set('scheduling-method', 'clerk', {silent: true});
                    button = view.ui.submit[0];

                    // Pre-Check that the view is in the right state
                    expect(button.classList.contains('hidden')).toBeTruthy();

                    view.onMethodChange();

                    expect(button.classList.contains('hidden')).toBeFalsy();
                    expect(button.innerText).toBe('Submit Request');
                });

                it('shows the direct submit button', function() {
                    var button;
                    
                    view.model.set('scheduling-method', 'direct', {silent: true});
                    button = view.ui.submit[0];

                    // Pre-Check that the view is in the right state
                    expect(button.classList.contains('hidden')).toBeTruthy();

                    view.onMethodChange();

                    expect(button.classList.contains('hidden')).toBeFalsy();
                    expect(button.innerText).toBe('Schedule Appointment');
                });
            });
        }

        describe('On Save Error', function() {
            beforeEach(function() {
                spyOn(view, 'onSaveSuccess').and.callFake(_.noop);
                spyOn(view, 'onServerError').and.callFake(_.noop);
            });

            it('catches fake errors and calls submit instead', function() {
                view.onSaveError(null, {status: 200}, null);
                expect(view.onSaveSuccess).toHaveBeenCalled();
                expect(view.onServerError).not.toHaveBeenCalled();
            });

            it('lets real errors through', function() {
                view.onSaveError(null, {status: 400}, null);
                expect(view.onSaveSuccess).not.toHaveBeenCalled();
                expect(view.onServerError).toHaveBeenCalled();
            });
        });


        describe('Creates the error region correctly', function() {
            var collection;
            var errors;

            beforeEach(function() {
                errors = [
                    {error: 'one'},
                    {error: 'two'},
                ];
                collection = new Backbone.Collection(errors);
            });

            it('draws the error region correctly', function() {
                var errorRegion;
                var errorView;
                var li;
                
                view.showErrorsRegion(collection);
                errorRegion = view.getRegion('error');
                errorView = errorRegion.currentView;
                li = errorView.$('li');

                expect(li[0].innerText).toBe(errors[0].error);
                expect(li[1].innerText).toBe(errors[1].error);
            });

            it('resets the errors region correctly', function() {
                var errorRegion;
                
                view.showErrorsRegion(collection);
                view.resetErrorRegion();
                errorRegion = view.getRegion('error');
                expect(errorRegion.currentView).toBeUndefined();
            });
        });

        describe('resetEmailRegion', function() {
            beforeEach(function() {
                view.showEmailRegion();
            });

            it('clears email email from the form', function() {
                var errorRegion = view.getRegion('email');
                
                expect(errorRegion.currentView).not.toBeUndefined();
                view.resetEmailRegion();
                expect(errorRegion.currentView).toBeUndefined();
            });
        });


        describe('On Submit', function() {
            beforeEach(function() {
                spyOn(view, 'saveForm').and.callFake(_.noop);
                spyOn(view, 'onFormErrors').and.callFake(_.noop);
            });

            it('submits the form when validation passes', function() {
                // Disable the validator for this test
                view.$('form').validate().settings.ignore = '*';
                view.onSubmit();
                expect(view.saveForm).toHaveBeenCalled();
            });
        });


        describe('on save', function() {
            var Model;
            var modelSpy;
            
            beforeEach(function() {
                Model = view.model.constructor;
                view.emailModel = new Backbone.Model();
                modelSpy = spyOn(Model.prototype, 'save').and.callFake(_.noop);
                spyOn(view, 'saveEmailPreferences').and.callFake(_.noop);
                spyOn(view, 'onSaveSuccess').and.callFake(_.noop);
                spyOn(view, 'onSaveError').and.callFake(_.noop);
                view.saveForm();
            });

            it('calls the model save function', function() {
                expect(modelSpy).toHaveBeenCalled();
                expect(view.saveEmailPreferences).toHaveBeenCalled();
            });

            it('sets the event listeners', function() {
                view.saveModel.trigger('save:success');
                view.saveModel.trigger('save:error');
                expect(view.onSaveSuccess).toHaveBeenCalled();
                expect(view.onSaveError).toHaveBeenCalled();
            });
        });


        describe('on server errors', function() {
            it('displays the server errors', function() {
                var data = {
                    responseJSON: {
                        errors: [
                            {error: 'one'},
                            {error: 'two'},
                        ],
                    },
                };
                var errorRegion;
                var errorView;
                var errors;
                
                view.onServerError(data);
                errorRegion = view.getRegion('error');
                errorView = errorRegion.currentView;
                errors = errorView.$('li');

                expect(errors[0].innerText).toBe(data.responseJSON.errors[0].error);
                expect(errors[1].innerText).toBe(data.responseJSON.errors[1].error);
            });
        });

        describe('onSaveSuccess', function() {
            beforeEach(function() {
                spyOn(view, 'disableShowConfirmNavigation').and.callFake(_.noop);
                spyOn(view, 'onAppointmentSaveSuccess').and.callFake(_.noop);
                spyOn(view, 'onRequestSaveSuccess').and.callFake(_.noop);
            });

            it('calls the appointment success function when appointment is saved', function() {
                view.model.set('scheduling-method', 'direct', {silent: true});
                view.onSaveSuccess();
                expect(view.onAppointmentSaveSuccess).toHaveBeenCalled();
            });

            it('calls the request success function when request is saved', function() {
                view.onSaveSuccess();
                expect(view.onRequestSaveSuccess).toHaveBeenCalled();
            });
        });


        describe('onAppointmentSaveSuccess', function() {
            beforeEach(function() {
                spyOn(view, 'navigateToDirectDetails').and.callFake(_.noop);
            });

            it('calls the navigate to details function with the correct params', function() {
                var details = new Backbone.Model();
                var model = new Backbone.Model();
                model.toAppointmentsModel = function() {
                    return details;
                };
                view.saveModel = model;
                view.onAppointmentSaveSuccess();

                expect(view.navigateToDirectDetails).toHaveBeenCalledWith(details);
            });
        });
    });
});
